/* The indicator shows the variation index of the price range calculated
 * on the lowest previous interval which is 2^n long. 
 * Variation Index shows if a trend
 * or a flat movement is prevailing in the time range or the range is behaving randomly.
 *
 * .. Dubovikov etc. - Dimension of the Minimal Cover 
 * and Local Analysis of Fractal Time Series.
 */
//+------------------------------------------------------------------+
//|                                                         iVAR.mq4 |
//|                                        (C)opyright  2008, Ilnur |
//|                                        http://www.metaquotes.net |
//+------------------------------------------------------------------+
//---- author of the indicator
#property copyright "(C)opyright  2008, Ilnur"
//---- link to the website of the author
#property link      "http://www.metaquotes.net"
//---- indicator version
#property version   "1.00"
#property description "Variation Index"
//---- drawing the indicator in a separate window
#property indicator_separate_window
//---- number of indicator buffers 1
#property indicator_buffers 1 
//---- only one plot is used
#property indicator_plots   1
//+-----------------------------------+
//|  Indicator drawing parameters     |
//+-----------------------------------+
//---- drawing indicator as a three-colored line
#property indicator_type1 DRAW_LINE
//---- red color is used as the color of the line
#property indicator_color1 Red
//---- indicator line is a solid one
#property indicator_style1 STYLE_SOLID
//---- indicator line width is equal to 2
#property indicator_width1 2
//---- displaying the signal line label
#property indicator_label1  "iVAR"
//+----------------------------------------------+
//| Horizontal levels display parameters         |
//+----------------------------------------------+
#property indicator_level1 0.5
#property indicator_levelcolor Blue
#property indicator_levelstyle STYLE_DASHDOTDOT
//+----------------------------------------------+
//|  Indicator input parameters                  |
//+----------------------------------------------+
input int N=5; // Indicator period  
//+----------------------------------------------+
//---- declaration of dynamic arrays that
// will be used as indicator buffers
double ExtLineBuffer[];

//---- declaration of a variable for storing the index value
int Num;
//---- declaration of a variable for storing the logarithm value
double MathLogX2;
//---- declaration of the integer variables for the start of data calculation
int StartBars;
//+------------------------------------------------------------------+
//|  searching index of the highest bar                              |
//+------------------------------------------------------------------+
int iHighest(const double &array[],// array for searching the maximum element index
             int count,// the number of the array elements (from a current bar to the index descending), 
             // along which the searching must be performed.
             int startPos //the initial bar index (shift relative to a current bar), 
             // the search for the greatest value begins from
             )
  {
//----
   int index=startPos;

//----checking correctness of the initial index
   if(startPos<0)
     {
      Print("Bad value in the iHighest function, startPos = ",startPos);
      return(0);
     }

//---- checking correctness of startPos value
   if(count<0) count=startPos;

   double max=array[startPos];

//---- searching for an index
   for(int i=startPos; i<startPos+count; i++)
     {
      if(array[i]>max)
        {
         index=i;
         max=array[i];
        }
     }
//---- returning of the greatest bar index
   return(index);
  }
//+------------------------------------------------------------------+
//|  Searching index of the lowest bar                               |
//+------------------------------------------------------------------+
int iLowest(const double &array[],// array for searching the minimum element index
            int count,            // the number of the array elements (from a current bar to the index descending) for searching           
            int startPos          // the initial bar index (shift relative to a current bar) for searching 
            )
  {
//----
   int index=startPos;

//---- checking correctness of the initial index
   if(startPos<0)
     {
      Print("Bad value in the iLowest function, startPos = ",startPos);
      return(0);
     }

//---- checking correctness of startPos value
   if(count<0) count=startPos;

   double min=array[startPos];

//---- searching for an index
   for(int i=startPos; i<startPos+count; i++)
     {
      if(array[i]<min)
        {
         index=i;
         min=array[i];
        }
     }
//---- returning of the lowest bar index
   return(index);
  }
//+------------------------------------------------------------------+
//| Custom indicator initialization function                         |
//+------------------------------------------------------------------+  
void OnInit()
  {
//---- initialization of variables of the start of data calculation
   StartBars=int(MathPow(2,N));
//---- initialization of variables 
   Num=N;
   MathLogX2=MathLog(2.0);
//---- set ExtLineBuffer dynamic array as indicator buffer
   SetIndexBuffer(0,ExtLineBuffer,INDICATOR_DATA);
//---- initializations of a variable for the indicator short name
   string shortname;
   StringConcatenate(shortname,"iVAR(",N,")");
//---- shifting the indicator horizontally by StartBars
   PlotIndexSetInteger(0,PLOT_SHIFT,StartBars);
//---- create label to display in Data Window
   PlotIndexSetString(0,PLOT_LABEL,shortname);
//---- restriction to draw empty values for the indicator
   PlotIndexSetDouble(0,PLOT_EMPTY_VALUE,EMPTY_VALUE);
//---- indexing elements in the buffer as timeseries
   ArraySetAsSeries(ExtLineBuffer,true);
//---- creating a name for displaying in a separate sub-window and in a tooltip
   IndicatorSetString(INDICATOR_SHORTNAME,shortname);
//---- determination of accuracy of displaying the indicator values
   IndicatorSetInteger(INDICATOR_DIGITS,_Digits);
//----
  }
//+------------------------------------------------------------------+
//| Custom indicator iteration function                              |
//+------------------------------------------------------------------+
int OnCalculate(const int rates_total,    // number of bars in history at the current tick
                const int prev_calculated,// number of bars calculated at previous call
                const datetime &time[],
                const double &open[],
                const double& high[],     // price array of maximums of price for the indicator calculation
                const double& low[],      // price array of minimums of price for the calculation of indicator
                const double &close[],
                const long &tick_volume[],
                const long &volume[],
                const int &spread[])
  {
//---- checking the number of bars to be enough for the calculation
   if(rates_total<StartBars) return(0);

//---- declarations of local variables 
   int limit,ihigh,ilow,nInterval;
   double Delta,Xc,Yc,Sx,Sy,Sxx,Sxy;

//---- calculation of the 'first' starting number for the bars recalculation loop
   if(prev_calculated>rates_total || prev_calculated<=0) // checking for the first start of the indicator calculation
      limit=rates_total-StartBars-1;                   // starting index for calculation of all bars
   else limit=rates_total-prev_calculated;               // starting index for calculation of new bars

//---- indexing elements in arrays as timeseries  
   ArraySetAsSeries(high,true);
   ArraySetAsSeries(low,true);

//---- main indicator calculation loop
   for(int bar=limit; bar>=0; bar--)
     {
      Sx = 0;
      Sy = 0;
      Sxx = 0;
      Sxy = 0;

      for(int iii=0; iii<=N; iii++)
        {
         Delta=0;
         nInterval=int(MathPow(2,N-iii));
         //---- summing the difference between the maximum and minimum prices on the interval
         for(int kkk=0; kkk<MathPow(2,iii); kkk++)
           {
            ihigh = iHighest(high,nInterval,bar+nInterval*kkk);
            ilow  = iLowest (low, nInterval,bar+nInterval*kkk);
            Delta+= high[ihigh]-low[ilow];
           }
         //---- calculate variation coordinates in a log-log scale
         Xc = (N-iii)*MathLogX2;
         Yc = MathLog(Delta);

         //---- collect data for searching regression line ratios using the least-squares method
         Sx += Xc;
         Sy += Yc;
         Sxx += Xc*Xc;
         Sxy += Xc*Yc;
        }
      //---- calculating the variation index (regression slope ratio)
      ExtLineBuffer[bar]=-(Sx*Sy-Num*Sxy)/(Sx*Sx-Num*Sxx);
     }
//----     
   return(rates_total);
  }
//+------------------------------------------------------------------+
